<?php
// Helper functions for the PrismaOS website

// Sanitize input data
function sanitize_input($data) {
    $data = trim($data);
    $data = stripslashes($data);
    $data = htmlspecialchars($data, ENT_QUOTES, 'UTF-8');
    return $data;
}

// Validate email
function validate_email($email) {
    return filter_var($email, FILTER_VALIDATE_EMAIL);
}

// Set flash message
function set_flash_message($message, $type = 'info') {
    $_SESSION['flash_message'] = $message;
    $_SESSION['flash_type'] = $type;
}

// Generate CSRF token
function generate_csrf_token() {
    if (!isset($_SESSION['csrf_token'])) {
        $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
    }
    return $_SESSION['csrf_token'];
}

// Verify CSRF token
function verify_csrf_token($token) {
    return isset($_SESSION['csrf_token']) && hash_equals($_SESSION['csrf_token'], $token);
}

// Contact form submission
function submit_contact_form($pdo, $data) {
    try {
        $sql = "INSERT INTO contact_submissions (name, email, subject, message, accessibility_needs) 
                VALUES (:name, :email, :subject, :message, :accessibility_needs)";
        
        $stmt = $pdo->prepare($sql);
        $result = $stmt->execute([
            ':name' => $data['name'],
            ':email' => $data['email'],
            ':subject' => $data['subject'],
            ':message' => $data['message'],
            ':accessibility_needs' => $data['accessibility_needs'] ?? ''
        ]);
        
        return $result;
    } catch (PDOException $e) {
        error_log("Contact form submission failed: " . $e->getMessage());
        return false;
    }
}

// Newsletter subscription
function subscribe_newsletter($pdo, $email) {
    try {
        $sql = "INSERT IGNORE INTO newsletter_subscriptions (email) VALUES (:email)";
        $stmt = $pdo->prepare($sql);
        $result = $stmt->execute([':email' => $email]);
        
        // Check if a row was actually inserted
        return $stmt->rowCount() > 0;
    } catch (PDOException $e) {
        error_log("Newsletter subscription failed: " . $e->getMessage());
        return false;
    }
}

// Get accessibility features data
function get_accessibility_features() {
    return [
        [
            'icon' => 'fas fa-deaf',
            'title' => 'For Deaf Users',
            'description' => 'Advanced closed captioning, visual notifications, sign language recognition, and customizable haptic feedback.'
        ],
        [
            'icon' => 'fas fa-blind',
            'title' => 'For Blind Users',
            'description' => 'Advanced text-to-speech, comprehensive voice commands, audio descriptions, and optimized gesture controls.'
        ],
        [
            'icon' => 'fas fa-wheelchair',
            'title' => 'Motor Accessibility',
            'description' => 'Customizable gesture controls, voice navigation, switch control support, and adaptive touch interfaces.'
        ],
        [
            'icon' => 'fas fa-brain',
            'title' => 'Cognitive Support',
            'description' => 'Simplified interfaces, memory aids, focus assistance, and customizable complexity levels.'
        ]
    ];
}

// Get technical specifications
function get_technical_specs() {
    return [
        [
            'category' => 'Operating System',
            'specs' => [
                'Base: Red Hat Enterprise Linux 9.x',
                'Kernel: Linux 6.x with accessibility patches',
                'Desktop: Custom accessibility-first environment',
                'Package Manager: DNF with accessibility repository'
            ]
        ],
        [
            'category' => 'Hardware Requirements',
            'specs' => [
                'CPU: ARMv8-A or x86_64 architecture',
                'RAM: Minimum 4GB, Recommended 8GB',
                'Storage: 32GB minimum, 64GB recommended',
                'Network: Wi-Fi 802.11ac, Bluetooth 5.0+'
            ]
        ],
        [
            'category' => 'Accessibility Hardware',
            'specs' => [
                'Screen Reader: Built-in TTS engine',
                'Haptic Feedback: Advanced vibration patterns',
                'Audio: Enhanced spatial audio processing',
                'Input: Switch control, eye tracking support'
            ]
        ]
    ];
}

// Get development roadmap
function get_roadmap_items() {
    return [
        [
            'phase' => 'Phase 1 - Foundation',
            'timeline' => 'Q1-Q2 2025',
            'status' => 'in-progress',
            'items' => [
                'Core accessibility framework development',
                'Basic screen reader integration',
                'Voice command system prototype',
                'Initial hardware compatibility testing'
            ]
        ],
        [
            'phase' => 'Phase 2 - Enhancement',
            'timeline' => 'Q3-Q4 2025',
            'status' => 'planned',
            'items' => [
                'Advanced gesture recognition',
                'AI-powered accessibility adaptation',
                'Comprehensive app ecosystem',
                'Beta testing with accessibility communities'
            ]
        ],
        [
            'phase' => 'Phase 3 - Launch',
            'timeline' => 'Q1-Q2 2026',
            'status' => 'planned',
            'items' => [
                'Public release preparation',
                'Manufacturing partnerships',
                'Global accessibility certification',
                'Community support infrastructure'
            ]
        ]
    ];
}
?>