<?php
// Handle contact form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $csrf_token = $_POST['csrf_token'] ?? '';
    
    if (!verify_csrf_token($csrf_token)) {
        set_flash_message('Security token mismatch. Please try again.', 'danger');
    } else {
        $name = sanitize_input($_POST['name'] ?? '');
        $email = sanitize_input($_POST['email'] ?? '');
        $subject = sanitize_input($_POST['subject'] ?? '');
        $message = sanitize_input($_POST['message'] ?? '');
        $accessibility_needs = sanitize_input($_POST['accessibility_needs'] ?? '');
        
        // Validate form data
        $errors = [];
        
        if (empty($name) || strlen($name) < 2) {
            $errors[] = 'Name must be at least 2 characters long.';
        }
        
        if (empty($email) || !validate_email($email)) {
            $errors[] = 'Please provide a valid email address.';
        }
        
        if (empty($subject) || strlen($subject) < 5) {
            $errors[] = 'Subject must be at least 5 characters long.';
        }
        
        if (empty($message) || strlen($message) < 10) {
            $errors[] = 'Message must be at least 10 characters long.';
        }
        
        if (empty($errors)) {
            $contact_data = [
                'name' => $name,
                'email' => $email,
                'subject' => $subject,
                'message' => $message,
                'accessibility_needs' => $accessibility_needs
            ];
            
            if (submit_contact_form($pdo, $contact_data)) {
                set_flash_message('Thank you for your message! We will get back to you within 24-48 hours.', 'success');
                // Redirect to prevent form resubmission
                header('Location: ?page=contact&sent=1');
                exit;
            } else {
                set_flash_message('Sorry, there was an error sending your message. Please try again.', 'danger');
            }
        } else {
            set_flash_message(implode(' ', $errors), 'danger');
        }
    }
}
?>

<section class="contact-section py-5">
    <div class="container">
        <div class="row">
            <div class="col-12 text-center mb-5">
                <h1 class="display-4 fw-bold">Contact PrismaOS Team</h1>
                <p class="lead text-muted">
                    Have questions about accessibility features? Need technical support? We're here to help.
                </p>
            </div>
        </div>
        
        <div class="row">
            <div class="col-lg-8 mx-auto">
                <?php if (isset($_GET['sent'])): ?>
                <div class="alert alert-success alert-dismissible fade show" role="alert">
                    <i class="fas fa-check-circle me-2" aria-hidden="true"></i>
                    <strong>Message Sent Successfully!</strong> We'll respond within 24-48 hours.
                    <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                </div>
                <?php endif; ?>
                
                <div class="contact-form-container bg-white p-4 border rounded-3 shadow-sm">
                    <form method="POST" novalidate class="needs-validation">
                        <input type="hidden" name="csrf_token" value="<?php echo generate_csrf_token(); ?>">
                        
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label for="name" class="form-label">
                                    Full Name <span class="text-danger">*</span>
                                </label>
                                <input type="text" class="form-control" id="name" name="name" 
                                       required minlength="2" maxlength="100"
                                       aria-describedby="name-help" aria-required="true"
                                       value="<?php echo isset($_POST['name']) ? htmlspecialchars($_POST['name']) : ''; ?>">
                                <div id="name-help" class="form-text">Enter your full name (2-100 characters)</div>
                                <div class="invalid-feedback">Please provide a valid name (2-100 characters).</div>
                            </div>
                            
                            <div class="col-md-6 mb-3">
                                <label for="email" class="form-label">
                                    Email Address <span class="text-danger">*</span>
                                </label>
                                <input type="email" class="form-control" id="email" name="email" 
                                       required aria-describedby="email-help" aria-required="true"
                                       value="<?php echo isset($_POST['email']) ? htmlspecialchars($_POST['email']) : ''; ?>">
                                <div id="email-help" class="form-text">We'll use this to respond to your inquiry</div>
                                <div class="invalid-feedback">Please provide a valid email address.</div>
                            </div>
                        </div>
                        
                        <div class="mb-3">
                            <label for="subject" class="form-label">
                                Subject <span class="text-danger">*</span>
                            </label>
                            <input type="text" class="form-control" id="subject" name="subject" 
                                   required minlength="5" maxlength="200"
                                   aria-describedby="subject-help" aria-required="true"
                                   value="<?php echo isset($_POST['subject']) ? htmlspecialchars($_POST['subject']) : ''; ?>">
                            <div id="subject-help" class="form-text">Brief description of your inquiry (5-200 characters)</div>
                            <div class="invalid-feedback">Please provide a subject (5-200 characters).</div>
                        </div>
                        
                        <div class="mb-3">
                            <label for="accessibility_needs" class="form-label">
                                Accessibility Requirements
                            </label>
                            <select class="form-select" id="accessibility_needs" name="accessibility_needs"
                                    aria-describedby="accessibility-help">
                                <option value="">No specific requirements</option>
                                <option value="screen_reader" <?php echo (isset($_POST['accessibility_needs']) && $_POST['accessibility_needs'] === 'screen_reader') ? 'selected' : ''; ?>>Screen reader user</option>
                                <option value="low_vision" <?php echo (isset($_POST['accessibility_needs']) && $_POST['accessibility_needs'] === 'low_vision') ? 'selected' : ''; ?>>Low vision/visual impairment</option>
                                <option value="hearing_impaired" <?php echo (isset($_POST['accessibility_needs']) && $_POST['accessibility_needs'] === 'hearing_impaired') ? 'selected' : ''; ?>>Deaf or hard of hearing</option>
                                <option value="motor_impairment" <?php echo (isset($_POST['accessibility_needs']) && $_POST['accessibility_needs'] === 'motor_impairment') ? 'selected' : ''; ?>>Motor/mobility impairment</option>
                                <option value="cognitive" <?php echo (isset($_POST['accessibility_needs']) && $_POST['accessibility_needs'] === 'cognitive') ? 'selected' : ''; ?>>Cognitive accessibility needs</option>
                                <option value="multiple" <?php echo (isset($_POST['accessibility_needs']) && $_POST['accessibility_needs'] === 'multiple') ? 'selected' : ''; ?>>Multiple accessibility needs</option>
                                <option value="other" <?php echo (isset($_POST['accessibility_needs']) && $_POST['accessibility_needs'] === 'other') ? 'selected' : ''; ?>>Other (please specify in message)</option>
                            </select>
                            <div id="accessibility-help" class="form-text">This helps us provide better support tailored to your needs</div>
                        </div>
                        
                        <div class="mb-4">
                            <label for="message" class="form-label">
                                Message <span class="text-danger">*</span>
                            </label>
                            <textarea class="form-control" id="message" name="message" rows="6" 
                                      required minlength="10" maxlength="2000"
                                      aria-describedby="message-help" aria-required="true"><?php echo isset($_POST['message']) ? htmlspecialchars($_POST['message']) : ''; ?></textarea>
                            <div id="message-help" class="form-text">Please provide details about your inquiry (10-2000 characters)</div>
                            <div class="invalid-feedback">Please provide a message (10-2000 characters).</div>
                        </div>
                        
                        <div class="text-center">
                            <button type="submit" class="btn btn-primary btn-lg">
                                <i class="fas fa-paper-plane me-2" aria-hidden="true"></i>
                                Send Message
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
        
        <!-- Contact Information -->
        <div class="row mt-5">
            <div class="col-md-4 text-center mb-4">
                <div class="contact-info-card p-4">
                    <i class="fas fa-envelope fa-3x text-primary mb-3" aria-hidden="true"></i>
                    <h3 class="h5 fw-bold">Email Support</h3>
                    <p class="text-muted">General inquiries and support</p>
                    <a href="mailto:support@prismaos.org" class="text-primary">support@prismaos.org</a>
                </div>
            </div>
            
            <div class="col-md-4 text-center mb-4">
                <div class="contact-info-card p-4">
                    <i class="fas fa-universal-access fa-3x text-primary mb-3" aria-hidden="true"></i>
                    <h3 class="h5 fw-bold">Accessibility Team</h3>
                    <p class="text-muted">Dedicated accessibility support</p>
                    <a href="mailto:accessibility@prismaos.org" class="text-primary">accessibility@prismaos.org</a>
                </div>
            </div>
            
            <div class="col-md-4 text-center mb-4">
                <div class="contact-info-card p-4">
                    <i class="fas fa-code fa-3x text-primary mb-3" aria-hidden="true"></i>
                    <h3 class="h5 fw-bold">Developer Support</h3>
                    <p class="text-muted">Technical and SDK questions</p>
                    <a href="mailto:developers@prismaos.org" class="text-primary">developers@prismaos.org</a>
                </div>
            </div>
        </div>
        
        <!-- Response Time Information -->
        <div class="row mt-4">
            <div class="col-12">
                <div class="alert alert-info" role="alert">
                    <i class="fas fa-info-circle me-2" aria-hidden="true"></i>
                    <strong>Response Times:</strong> We typically respond to inquiries within 24-48 hours during business days. 
                    Accessibility-related questions receive priority handling.
                </div>
            </div>
        </div>
    </div>
</section>

<script>
// Bootstrap form validation
(function() {
    'use strict';
    window.addEventListener('load', function() {
        var forms = document.getElementsByClassName('needs-validation');
        var validation = Array.prototype.filter.call(forms, function(form) {
            form.addEventListener('submit', function(event) {
                if (form.checkValidity() === false) {
                    event.preventDefault();
                    event.stopPropagation();
                }
                form.classList.add('was-validated');
            }, false);
        });
    }, false);
})();
</script>